
#---------------------------------------------------------------------------------------#
#---------------------------------------------------------------------------------------#
# Nicholas Garvin 16 Aug 2018. Contact garvinn@rba.gov.au or nick.j.garvin@gmail.com
#    for enquiries or comments.
#
# The following code gives a demonstration of the algorithm by running it on randomly 
#   generated 'dummy' transactions data.
#
# There are three sections:
# 1) Sets the parameters for running the algorithm (e.g. maturity cap etc), generates
#    the dummy transactions data, runs the algorithm (which saves the output externally),
#    and reads the output back in. Requires 'repo detection algorithm FUNCTIONS.R'
# 2) Analyses the dummy algorithm output by generating a column of Tables 1 and 2. Also
#    turns the algorithm output into a perusable dataframe 'repo.data'. Requires
#    'Garvin (2018) tables etc FUNCTIONS.R'.
# 3) Exports the dummy transactions data and algorithm output for readers wishing to 
#    better understand the 'Garvin (2018) tables etc FUNCTIONS.R' script. The final part
#    of the 'Garvin (2018) tables etc RUN CODE.R' script reads these exported objects and 
#    formats them to resemble confidential objects used in 'Garvin (2018) tables etc
#    FUNCTIONS.R'.
#---------------------------------------------------------------------------------------#


rm(list=ls())  # Clear the environment
options(error=recover)  # Change error message response to aid debugging

# Change this line to directory containing 'repo detection algorithm FUNCTIONS.R'
WDalg <- 'Z:/Writing up results/Algorithm RDP/Online materials'
# and this line to the directory containing 'Garvin (2018) tables etc FUNCTIONS.R'
WDfiles <- 'Z:/Writing up results/Algorithm RDP/Online materials'
# and this line to the directory where output files should be saved
WDout <- 'Z:/Writing up results/Algorithm RDP/Online materials/outputs - IGNORE'


### Section 1: Running the algorithm
###-----------------------------------------------------------------------------------###
### THIS SECTION OF CODE RUNS THE ALGORITHM, EXPORTS THE OUTPUT AS A .R FILE, AND LOADS
###   IT BACK INTO THE GLOBAL ENVIRONMENT AS A LIST LABELLED 'y' WITH STRUCTURE:
# [[1]] intraday repos (list of 2-element vectors)
# [[2]] transactions data
# [[3]] prlctn.2tr (list of 2 matrices)
#   [[1]] prlctn.Mu (2col matrix of unique 2tr repos)
#   [[2]] prlctn.Mm (2col matrix of nonunique 2tr repos)
# [[4]] alloutput 
#   [[1]] list of vectors containing multiple-transaction repos
#   [[2]] 2col matrix. One row per candidate vector. Col1: length. Col2: 1=mtrx det,
#         -1=iter det, 0=no detection.

# Arguments (e.g. parameters etc) for running the algorithm. Described in more detail in
#   'repo detection algorithm FUNCTIONS.R'
mat.cap <- 14                 # Maturity cap in days
interest.bounds <- c(0, 0.1)  # Interest bounds (e.g. 0.1 is 10%)     
tr.cap <- 6                   # Transactions cap
WD <- WDout                   # Location for saving the algorithm output.
matrix.max <- 22              # Max candidate-vector length for matrix algebra method.
iter.max <- 45                # Max candidate-vector length for iterative method.
no.intraday <- FALSE          # Skip intraday-repo detection and removal?
no.multtrns <- FALSE          # Look only for two-transaction repos?
intsprdpos <- TRUE            # Set to FALSE for placebo tests
flnm.pref <- 'DUMMY'          # Prefix for the filename of exported output.

# Read in algorithm functions
setwd(WDalg)
source('repo detection algorithm FUNCTIONS.R')

# Create a dummy transactions dataset that the algorithm can be run on as an example.
dummy.trns.data.F <- function(obs, days, banks, secs) {
  # Constructs a dummy transactions dataset that works with the repo-detection 
  #   algorithm.
  #
  # Function arguments are each scalar integers, specifying the number of observations,
  #   days, counterparties and security ISINs to occur in the dataset.
  
  ST <- sample(1:days, obs, replace=TRUE) + runif(obs, 0.3, 0.8)
  Snd <- sample(1111:(1111+banks), obs*2, replace=TRUE)
  Rec <- sample(1111:(1111+banks), obs*2, replace=TRUE)
  # Ensure that Snd does not equal Rec
  notmatched <- Rec != Snd
  Snd <- Snd[notmatched][1:obs]
  Rec <- Rec[notmatched][1:obs]
  ISIN <- do.call(paste0, expand.grid(LETTERS, LETTERS))
  ISIN <- paste0(ISIN, ISIN, ISIN, ISIN, ISIN, ISIN)
  ISIN <- sample(ISIN[1:secs], obs, replace=TRUE)
  FV <- -sample(10:1000, obs, replace=TRUE)*1e5
  Cons <- -FV + runif(obs, min=-1e5, max=1e5)
  y <- data.frame(ST, Snd, Rec, ISIN, FV, Cons)
  # DATA MUST BE ORDERED BY SETTLEMENT TIME
  y <- y[order(y$ST), ]
  y
}
trns.data <- dummy.trns.data.F(obs=1e4, days=40, banks=10, secs=10)
# Display the first few rows of the dataset.
head(trns.data)

# Alternatively, have your transactions data in the same format as the dummy data above
#   and saved in the directory specified above as a csv file. Uncomment the following 
#   line and change the transactions filename as appropriate.
# trns.data <- read.csv('transactions filename.csv)

# Run the algorithm on the transactions data. The function saves the output into 
#   directory specified above (rather than generating it in this R environment). A more
#   detailed description of the function is in 'repo detection algorithm FUNCTIONS.R'.
repo.alg.F(trns.data, mat.cap, interest.bounds, tr.cap, WD, matrix.max,    
           iter.max, no.intraday, no.multtrns, intsprdpos, flnm.pref)

# Another example that includes only necessary arguments (i.e. without defaults)
repo.alg.F(trns.data, mat.cap, interest.bounds, tr.cap, WD, flnm.pref='DUMMY2')

# Read the algorithm output into the global environment. 
filename <- paste0(flnm.pref, '_algoutput_mat', mat.cap, '_int', interest.bounds[1], '-', 
                   interest.bounds[2], '_trcap', tr.cap, '.Rdata')
load(filename)
# Rename the algorithm output
alg.output <- y
remove(y)
###-----------------------------------------------------------------------------------###



### Section 2: Looking at the algorithm output
###-----------------------------------------------------------------------------------###
### THIS SECTION OF CODE TURNS THE ALGORITHM OUTPUT INTO A MORE ILLUSTRATIVE DATASET 
###   AND REPLICATES SOME OF THE TABLES IN Garvin (2018).

# Read in functions for generating tables
setwd(WDfiles)
source('Garvin (2018) tables etc FUNCTIONS.R')

# Generate a column of Table 1
stats.table.F(years=1, data.objects=list(list(alg.output, NULL), NULL), year=1, 
              whichtable=1)

# Generate a column of Table 2
stats.table.F(years=1, data.objects=list(list(alg.output, NULL), NULL), year=1, 
              whichtable=2)

# Turn the algorithm output into a dataset. 
repo.data <- repo.df.F(year=1, data.objects=list(list(alg.output, NULL), NULL), 
                       years=1, neg=FALSE, allvars=FALSE) 
head(repo.data)
###-----------------------------------------------------------------------------------###





### Section 3: Export objects for interpreting 'Garvin (2018) table etc FUNCTIONS.R'
###-----------------------------------------------------------------------------------###
### Export some of the objects to be read by 'Garvin (2018) table etc RUN CODE.R'
setwd(WDout)
save(trns.data, alg.output, list=c('trns.data', 'alg.output'), 
     file='dummy transaction and repo data.Rdata')
###-----------------------------------------------------------------------------------###




